import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay_merchant/data/model/invoice/invoice_history_response_model.dart';
import 'package:viserpay_merchant/data/model/usercheck/user_check_response_modal.dart';

import '../../../core/helper/string_format_helper.dart';
import '../../../core/route/route.dart';
import '../../../core/utils/my_strings.dart';
import '../../../view/components/bottom-sheet/custom_bottom_sheet.dart';
import '../../../view/components/snack_bar/show_custom_snackbar.dart';
import '../../../view/screens/invoice/create_invoice/widget/create_invoice_bottom_sheet.dart';
import '../../../view/screens/invoice/create_invoice/widget/invoice_items.dart';
import '../../model/authorization/authorization_response_model.dart';
import '../../model/global/response_model/response_model.dart';
import '../../model/invoice/create_invoice_response_model.dart';
import '../../model/invoice/invoice_items_model.dart';
import '../../repo/invoice/create_invoice_repo.dart';

class CreateInvoiceController extends GetxController {
  CreateInvoiceRepo createInvoiceRepo;
  CreateInvoiceController({required this.createInvoiceRepo});

  final formKey = GlobalKey<FormState>();

  List<InvoiceItemsModel> invoiceItemList = [];
  PaymentCharge paymentChargeData = PaymentCharge();
  TextEditingController invoiceToController = TextEditingController();

  TextEditingController itemController = TextEditingController();
  TextEditingController amountController = TextEditingController();

  void increaseNumberField() {
    invoiceItemList.add(InvoiceItemsModel(itemNameController: TextEditingController(), amountController: TextEditingController()));
    update();
  }

  void decreaseNumberField(int index) {
    invoiceItemList.removeAt(index);
    calculateInvoiceAmount();
    update();
  }

  String totalInvoiceAmount = '0.00';
  String charge = '0.00';
  String getableText = '0.00';

  void calculateInvoiceAmount() {
    double totalAmount = 0;

    double firstInvoiceAmount = double.tryParse(amountController.text.toString()) ?? 0;
    totalAmount = totalAmount + firstInvoiceAmount;

    for (var invoice in invoiceItemList) {
      double invoiceAmount = double.tryParse(invoice.amountController.text) ?? 0;
      totalAmount = totalAmount + invoiceAmount;
    }

    totalInvoiceAmount = StringConverter.formatNumber(totalAmount.toString(), precision: 2);
    double cap = double.tryParse(paymentChargeData.cap ?? "0") ?? 0;
    double percent = double.tryParse(paymentChargeData.merchantPaymentPercentCharge ?? "0") ?? 0;
    double percentCharge = totalAmount * percent / 100;

    double fixedCharge = double.tryParse(paymentChargeData.merchantPaymentFixedCharge ?? "0") ?? 0;

    double totalCharge = percentCharge + fixedCharge;

    if (cap != -1.0 && cap != 1 && totalCharge > cap) {
      totalCharge = cap;
    }

    charge = StringConverter.formatNumber('$totalCharge', precision: 2);

    double payable = totalAmount - totalCharge;
    getableText = StringConverter.formatNumber(payable.toString(), precision: 2);

    update();
  }

  loadData(PaymentCharge paymentCharge) async {
    invoiceToController.text = "";
    invoiceItemList.clear();
    itemController.text = '';
    amountController.text = '';
    paymentChargeData = paymentCharge;
    charge = "0.00";
    totalInvoiceAmount = "0.00";
    getableText = "0.00";
    userExist = false;
    update();
  }

  Future<void> confirmInvoice() async {
    String invoiceTo = invoiceToController.text.toString();

    if (invoiceTo.isEmpty) {
      CustomSnackBar.error(errorList: [MyStrings.invoiceFieldErrorMsg]);
      return;
    }

    CustomBottomSheet(child: const CreateInvoicePreviewBottomSheet()).customBottomSheet(Get.context!);
  }

  bool isSubmitLoading = false;
  Future<void> submitInvoice() async {
    String invoiceTo = invoiceToController.text.toString();

    if (invoiceTo.isEmpty) {
      CustomSnackBar.error(errorList: [MyStrings.invoiceFieldErrorMsg]);
      return;
    }

    String firstInvoice = itemController.text.toString();
    String firstInvoiceAmount = amountController.text.toString();

    if (firstInvoice.isEmpty) {
      CustomSnackBar.error(errorList: [MyStrings.invoiceItemNameErrorMsg]);
      return;
    }
    if (firstInvoiceAmount.isEmpty) {
      CustomSnackBar.error(errorList: [MyStrings.invoiceAmountErrorMsg]);
      return;
    }

    isSubmitLoading = true;
    update();

    ResponseModel responseModel = await createInvoiceRepo.createInvoice(invoiceTo, firstInvoice, firstInvoiceAmount, invoiceItemList);
    try {
      if (responseModel.statusCode == 200) {
        AuthorizationResponseModel model = AuthorizationResponseModel.fromJson(jsonDecode(responseModel.responseJson));
        if (model.status.toString().toLowerCase() == MyStrings.success.toLowerCase()) {
          Get.offAndToNamed(RouteHelper.invoiceScreen);
          CustomSnackBar.success(successList: model.message?.success ?? [MyStrings.requestSuccess]);
        } else {
          CustomSnackBar.error(errorList: model.message?.error ?? [MyStrings.somethingWentWrong]);
        }
      } else {
        CustomSnackBar.error(errorList: [responseModel.message]);
        return;
      }
    } catch (e) {
      isSubmitLoading = false;
      update();
    }

    isSubmitLoading = false;
    update();
  }

  bool userExist = false;
  UserCheckResponseModal? existUserModal;
  Future<void> checkUserExist(String userName) async {
    userExist = false;
    existUserModal = null;
    update();
    try {
      ResponseModel responseModel = await createInvoiceRepo.checkUser(usernameOrmobile: userName.replaceAll('+', '').replaceAll("-", "".replaceAll(" ", "")));
      if (responseModel.statusCode == 200) {
        existUserModal = UserCheckResponseModal.fromJson(jsonDecode(responseModel.responseJson));
        if (existUserModal?.status == "success") {
          userExist = true;
          existUserModal = existUserModal;
          update();
        } else {
          userExist = false;
          existUserModal = null;
          update();
        }
      } else {
        CustomSnackBar.error(errorList: [responseModel.message]);
      }
    } catch (e) {
      userExist = false;
      existUserModal = null;
      update();
    }
  }
}
